# 🤖 Crypto AI Trader - MVP

Sistema di trading e arbitraggio crypto gestito da AI (Claude), con supporto paper trading e possibilità di evoluzione a trading reale.

## 🎯 Features MVP

- ✅ **Market Data Real-Time**: WebSocket multipli exchange (Binance, Coinbase, Kraken)
- ✅ **AI Decision Engine**: Claude Sonnet 4.5 per analisi e decisioni trading
- ✅ **Paper Trading**: Simulazione completa con fee/slippage/risk management
- ✅ **Portfolio Management**: Tracking posizioni, PnL, equity curve
- ✅ **Dashboard Web**: Grafici real-time, ordini, AI reasoning logs
- ✅ **Technical Indicators**: RSI, EMA, Volume, Volatility
- ✅ **Risk Management**: Position sizing, stop-loss, max drawdown

## 🚀 Quick Start (5 minuti)

### Prerequisiti

- Docker & Docker Compose
- Chiave API Anthropic (Claude)
- (Opzionale) Chiavi API exchange per live data

### 1. Clone e Setup

```bash
# Se hai il progetto come zip, estrailo
cd crypto-ai-trader

# Copia e configura le variabili d'ambiente
cp .env.example .env
nano .env  # Inserisci la tua ANTHROPIC_API_KEY
```

### 2. Avvia i Container

```bash
docker-compose up -d
```

Questo comando:
- Crea il database PostgreSQL con TimescaleDB
- Avvia il backend FastAPI
- Avvia il frontend React
- Inizializza le tabelle

### 3. Accedi alla Dashboard

Apri il browser: **http://localhost:3000**

### 4. Verifica che Funzioni

```bash
# Logs backend
docker-compose logs -f backend

# Logs database
docker-compose logs db

# Status di tutti i servizi
docker-compose ps
```

## 📊 Architettura

```
┌─────────────────────────────────────────────┐
│         Frontend (React + Vite)              │
│  http://localhost:3000                       │
└──────────────────┬──────────────────────────┘
                   │ WebSocket + REST API
┌──────────────────┴──────────────────────────┐
│       Backend (FastAPI + Python)             │
│  http://localhost:8000                       │
│  ┌──────────────┐ ┌──────────────┐          │
│  │ Market Data  │ │ AI Strategy  │          │
│  │  (CCXT Pro)  │ │   (Claude)   │          │
│  └──────┬───────┘ └──────┬───────┘          │
│         │                 │                  │
│  ┌──────┴─────────────────┴───────┐         │
│  │    Execution Simulator          │         │
│  │    Portfolio Manager            │         │
│  └─────────────┬───────────────────┘         │
└────────────────┼─────────────────────────────┘
                 │
┌────────────────┴─────────────────────────────┐
│  PostgreSQL + TimescaleDB                    │
│  localhost:5432                              │
└──────────────────────────────────────────────┘
```

## 🔧 Configurazione (.env)

```bash
# AI Provider
ANTHROPIC_API_KEY=sk-ant-xxx  # OBBLIGATORIO

# Database
DATABASE_URL=postgresql://trader:trader123@db:5432/crypto_trading

# Trading Settings
PAPER_TRADING=true
INITIAL_BALANCE_USDT=10000
MAX_POSITION_SIZE_PCT=0.30  # 30% del portfolio per trade
RISK_PER_TRADE_PCT=0.02     # 2% max loss per trade

# Exchange API Keys (opzionali per MVP - usa dati pubblici)
BINANCE_API_KEY=
BINANCE_SECRET=
COINBASE_API_KEY=
COINBASE_SECRET=

# AI Strategy Settings
AI_CHECK_INTERVAL_SECONDS=60        # Ogni quanto analizza
AI_MIN_CONFIDENCE=0.70              # Confidence minima per trade
AI_MAX_TRADES_PER_DAY=20

# Exchanges da monitorare
ACTIVE_EXCHANGES=binance,coinbase,kraken
DEFAULT_SYMBOLS=BTC/USDT,ETH/USDT

# Simulation Settings
MAKER_FEE=0.001      # 0.1%
TAKER_FEE=0.001      # 0.1%
SLIPPAGE_PCT=0.0005  # 0.05%
```

## 📁 Struttura Progetto

```
crypto-ai-trader/
├── backend/
│   ├── app/
│   │   ├── main.py              # FastAPI app
│   │   ├── config.py            # Settings
│   │   ├── models.py            # DB models
│   │   ├── database.py          # DB connection
│   │   ├── services/
│   │   │   ├── market_data.py   # CCXT + WebSocket
│   │   │   ├── ai_strategy.py   # Claude integration
│   │   │   ├── execution.py     # Order simulator
│   │   │   └── portfolio.py     # Portfolio state
│   │   ├── api/
│   │   │   ├── routes.py        # REST endpoints
│   │   │   └── websocket.py     # WS real-time
│   │   └── utils/
│   │       └── indicators.py    # RSI, EMA, etc.
│   ├── requirements.txt
│   └── Dockerfile
├── frontend/
│   ├── src/
│   │   ├── components/          # React components
│   │   ├── hooks/               # Custom hooks
│   │   ├── App.tsx
│   │   └── main.tsx
│   ├── package.json
│   └── Dockerfile
├── database/
│   └── init.sql                 # Schema iniziale
├── docker-compose.yml
├── .env.example
└── README.md
```

## 🎮 Uso del Sistema

### Dashboard Principale

La dashboard mostra:

1. **Portfolio Overview**: Equity, PnL, posizioni aperte
2. **Live Charts**: Candlestick charts con TradingView
3. **AI Status**: Ultimo segnale, confidence, reasoning
4. **Trade Log**: Storico ordini eseguiti
5. **Controls**: Start/Stop AI, ordini manuali

### API Endpoints

Backend espone API REST su `http://localhost:8000`:

```bash
# Documentazione interattiva
http://localhost:8000/docs

# Endpoints principali
GET  /api/portfolio          # Stato portfolio
GET  /api/trades             # Storico trade
GET  /api/positions          # Posizioni aperte
POST /api/strategy/analyze   # Trigger analisi AI
POST /api/orders/manual      # Ordine manuale
GET  /api/market/ticker/{symbol}  # Ultimo prezzo

# WebSocket
WS   /ws  # Stream real-time (tickers, signals, fills)
```

### WebSocket Events

Il frontend riceve eventi real-time:

```json
// Market update
{"type": "ticker", "symbol": "BTC/USDT", "price": 63420.5}

// AI signal
{
  "type": "ai_signal",
  "timestamp": 1730697600000,
  "symbol": "BTC/USDT",
  "action": "BUY",
  "confidence": 0.78,
  "reasoning": "RSI oversold + volume spike, bullish divergence",
  "size_usdt": 500
}

// Order fill
{"type": "order_filled", "order_id": "abc123", "symbol": "BTC/USDT"}

// Portfolio update
{"type": "portfolio", "equity": 10234.56, "pnl_pct": 2.34}
```

## 🧪 Testing

### Test Backend

```bash
# Entra nel container
docker-compose exec backend bash

# Test connessione exchange
python -c "import ccxt; print(ccxt.binance().fetch_ticker('BTC/USDT'))"

# Test AI
python -m app.services.ai_strategy
```

### Test Database

```bash
# Accedi al DB
docker-compose exec db psql -U trader -d crypto_trading

# Query esempio
SELECT * FROM trades ORDER BY timestamp DESC LIMIT 10;
SELECT * FROM portfolio_snapshots ORDER BY timestamp DESC LIMIT 5;
```

## 🔄 Workflow Tipico

1. **Sistema si avvia** → Connette exchange, inizializza portfolio
2. **Market data stream** → Riceve ticker/candles real-time
3. **AI analysis loop** (ogni 60s):
   - Calcola indicatori tecnici
   - Costruisce context per Claude
   - Claude analizza e decide (BUY/SELL/HOLD)
   - Log decision con reasoning
4. **Se signal valido**:
   - Verifica risk management
   - Invia ordine a Execution Simulator
   - Simula fill con fee/slippage
   - Aggiorna portfolio
5. **Frontend** riceve update via WebSocket → mostra tutto real-time

## 🚀 Evoluzione a Produzione

### Fase 1: Backtest (dopo MVP)
- Replay dati storici
- Ottimizzazione parametri
- A/B test strategie

### Fase 2: Live Trading
- Cambia `PAPER_TRADING=false` in .env
- Aggiungi chiavi API exchange reali
- Execution Simulator → Exchange Real API
- **⚠️ ATTENZIONE: Trading reale comporta rischi!**

### Fase 3: Features Avanzate
- Arbitraggio triangolare
- Multi-strategy orchestration
- Leverage trading (futures)
- Advanced risk management
- Performance analytics

## 🛠️ Troubleshooting

### Backend non si avvia

```bash
# Verifica logs
docker-compose logs backend

# Ricostruisci immagine
docker-compose build --no-cache backend
docker-compose up -d backend
```

### Database connection error

```bash
# Verifica che il DB sia up
docker-compose ps db

# Reset database
docker-compose down -v
docker-compose up -d db
# Aspetta 10 secondi
docker-compose up -d backend
```

### Frontend non carica

```bash
# Verifica che backend sia reachable
curl http://localhost:8000/api/health

# Rebuild frontend
docker-compose build --no-cache frontend
docker-compose up -d frontend
```

### CCXT errori di connessione

- Controlla firewall/proxy
- Alcuni exchange richiedono API keys anche per dati pubblici
- Verifica rate limits

## 📚 Documentazione Utile

- **FastAPI**: https://fastapi.tiangolo.com/
- **CCXT**: https://docs.ccxt.com/
- **Anthropic API**: https://docs.anthropic.com/
- **TimescaleDB**: https://docs.timescale.com/
- **TradingView Charts**: https://www.tradingview.com/lightweight-charts/

## ⚠️ Disclaimer

Questo sistema è fornito "as-is" solo per scopi educativi e di ricerca. 

- **Non è un consiglio finanziario**
- Il trading crypto comporta rischi significativi
- Possibilità di perdere tutto il capitale investito
- Testare sempre in paper trading prima di usare fondi reali
- L'AI può fare errori - supervisione umana necessaria

## 📝 License

MIT License - Vedi LICENSE file per dettagli

---

**🎯 Pronto per iniziare? Segui gli step di Quick Start!**

Per domande o problemi: apri una Issue su GitHub (se condiviso) o contatta il maintainer.
