"""
REST API Routes
"""
from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from typing import List, Optional
from datetime import datetime, timedelta
from app.database import get_db
from app.models import Trade, Position, AISignal, PortfolioSnapshot
from app.services.portfolio import portfolio_service
from app.services.market_data import market_data_service
from app.services.ai_strategy import ai_strategy_service
from app.services.execution import execution_simulator
from app.utils.indicators import calculate_indicators
import logging

logger = logging.getLogger(__name__)

router = APIRouter(prefix="/api")


@router.get("/health")
async def health_check():
    """Health check endpoint"""
    return {"status": "healthy", "timestamp": datetime.now().isoformat()}


@router.get("/portfolio")
async def get_portfolio(db: Session = Depends(get_db)):
    """Get current portfolio state"""
    try:
        # Get current prices
        current_prices = {}
        for symbol in ['BTC/USDT', 'ETH/USDT']:
            ticker = await market_data_service.fetch_ticker('binance', symbol)
            if ticker:
                current_prices[symbol] = ticker['last']
        
        state = portfolio_service.get_state(db, current_prices)
        return state
        
    except Exception as e:
        logger.error(f"Error getting portfolio: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/positions")
async def get_positions(db: Session = Depends(get_db)):
    """Get all open positions"""
    try:
        positions = db.query(Position).filter(Position.quantity > 0).all()
        
        return [
            {
                'id': p.id,
                'symbol': p.symbol,
                'exchange': p.exchange,
                'side': p.side,
                'quantity': float(p.quantity),
                'entry_price': float(p.entry_price),
                'current_price': float(p.current_price) if p.current_price else None,
                'pnl_unrealized': float(p.pnl_unrealized),
                'pnl_realized': float(p.pnl_realized),
                'opened_at': p.opened_at.isoformat(),
            }
            for p in positions
        ]
    except Exception as e:
        logger.error(f"Error getting positions: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/trades")
async def get_trades(
    limit: int = 50,
    symbol: Optional[str] = None,
    db: Session = Depends(get_db)
):
    """Get trade history"""
    try:
        query = db.query(Trade).order_by(Trade.timestamp.desc())
        
        if symbol:
            query = query.filter(Trade.symbol == symbol)
        
        trades = query.limit(limit).all()
        
        return [
            {
                'id': t.id,
                'timestamp': t.timestamp.isoformat(),
                'exchange': t.exchange,
                'symbol': t.symbol,
                'side': t.side,
                'type': t.type,
                'quantity': float(t.quantity),
                'price': float(t.price),
                'fee': float(t.fee),
                'pnl_realized': float(t.pnl_realized) if t.pnl_realized else None,
            }
            for t in trades
        ]
    except Exception as e:
        logger.error(f"Error getting trades: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/signals")
async def get_ai_signals(
    limit: int = 20,
    symbol: Optional[str] = None,
    db: Session = Depends(get_db)
):
    """Get AI signal history"""
    try:
        query = db.query(AISignal).order_by(AISignal.timestamp.desc())
        
        if symbol:
            query = query.filter(AISignal.symbol == symbol)
        
        signals = query.limit(limit).all()
        
        return [
            {
                'id': s.id,
                'timestamp': s.timestamp.isoformat(),
                'symbol': s.symbol,
                'action': s.action,
                'confidence': float(s.confidence),
                'size_usdt': float(s.size_usdt) if s.size_usdt else None,
                'reasoning': s.reasoning,
                'executed': s.executed,
            }
            for s in signals
        ]
    except Exception as e:
        logger.error(f"Error getting signals: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.post("/strategy/analyze")
async def trigger_analysis(
    symbol: str = "BTC/USDT",
    db: Session = Depends(get_db)
):
    """Manually trigger AI analysis for a symbol"""
    try:
        # Get market data
        candles = await market_data_service.fetch_ohlcv(
            'binance', symbol, '5m', limit=100
        )
        
        if not candles:
            raise HTTPException(status_code=404, detail="No market data available")
        
        # Calculate indicators
        indicators = calculate_indicators(candles)
        
        # Get current price
        ticker = await market_data_service.fetch_ticker('binance', symbol)
        current_price = ticker['last'] if ticker else candles[-1][4]
        
        # Get portfolio state
        current_prices = {symbol: current_price}
        portfolio = portfolio_service.get_state(db, current_prices)
        
        # Prepare market data
        market_data = {
            'current_price': current_price,
            'indicators': indicators,
            'last_candles': candles[-10:],  # Last 10 candles
        }
        
        # Constraints
        constraints = {
            'max_position_size': portfolio['max_position_size_usdt'],
            'max_open_positions': 5,
            'min_confidence': 0.70,
        }
        
        # Get AI decision
        decision = await ai_strategy_service.analyze(
            symbol=symbol,
            market_data=market_data,
            portfolio=portfolio,
            constraints=constraints
        )
        
        if not decision:
            raise HTTPException(status_code=500, detail="AI analysis failed")
        
        # Save signal
        signal = AISignal(
            symbol=symbol,
            action=decision['action'],
            confidence=decision['confidence'],
            size_usdt=decision.get('size_usdt'),
            reasoning=decision['reasoning'],
            context=decision['context'],
            stop_loss=decision.get('stop_loss'),
            take_profit=decision.get('take_profit'),
        )
        db.add(signal)
        db.commit()
        
        return {
            'signal_id': signal.id,
            'decision': decision,
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in analysis: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.post("/orders/manual")
async def create_manual_order(
    symbol: str,
    side: str,
    quantity: float,
    db: Session = Depends(get_db)
):
    """Create a manual order (for testing)"""
    try:
        # Get current price
        ticker = await market_data_service.fetch_ticker('binance', symbol)
        if not ticker:
            raise HTTPException(status_code=404, detail="Symbol not found")
        
        current_price = ticker['last']
        
        # Execute order
        result = await execution_simulator.execute_market_order(
            db=db,
            exchange='binance',
            symbol=symbol,
            side=side.upper(),
            quantity=quantity,
            current_price=current_price
        )
        
        if not result:
            raise HTTPException(status_code=500, detail="Order execution failed")
        
        return result
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error creating order: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/market/ticker/{symbol}")
async def get_ticker(symbol: str):
    """Get current ticker for a symbol"""
    try:
        ticker = await market_data_service.fetch_ticker('binance', symbol)
        
        if not ticker:
            raise HTTPException(status_code=404, detail="Symbol not found")
        
        return ticker
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error getting ticker: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/stats/performance")
async def get_performance_stats(
    days: int = 7,
    db: Session = Depends(get_db)
):
    """Get performance statistics"""
    try:
        # Get snapshots from last N days
        since = datetime.now() - timedelta(days=days)
        snapshots = db.query(PortfolioSnapshot).filter(
            PortfolioSnapshot.timestamp >= since
        ).order_by(PortfolioSnapshot.timestamp.asc()).all()
        
        if not snapshots:
            return {
                'equity_curve': [],
                'total_trades': 0,
                'win_rate': 0,
                'total_pnl': 0,
            }
        
        # Build equity curve
        equity_curve = [
            {
                'timestamp': s.timestamp.isoformat(),
                'equity': float(s.equity_usdt),
                'pnl_pct': float(s.pnl_pct),
            }
            for s in snapshots
        ]
        
        # Calculate statistics
        all_trades = db.query(Trade).all()
        winning_trades = [t for t in all_trades if t.pnl_realized and t.pnl_realized > 0]
        
        return {
            'equity_curve': equity_curve,
            'total_trades': len(all_trades),
            'winning_trades': len(winning_trades),
            'win_rate': len(winning_trades) / len(all_trades) * 100 if all_trades else 0,
            'total_pnl': float(snapshots[-1].pnl_total) if snapshots else 0,
            'current_equity': float(snapshots[-1].equity_usdt) if snapshots else 0,
        }
        
    except Exception as e:
        logger.error(f"Error getting stats: {e}")
        raise HTTPException(status_code=500, detail=str(e))
