"""
AI Strategy Service - Claude integration for trading decisions
"""
import json
from typing import Dict, Optional
from datetime import datetime
from anthropic import Anthropic
from app.config import settings
import logging

logger = logging.getLogger(__name__)


class AIStrategyService:
    """
    AI-powered trading strategy using Claude
    """
    
    def __init__(self):
        self.client = Anthropic(api_key=settings.anthropic_api_key)
        self.model = settings.ai_model
        
        # System prompt for the AI trader
        self.system_prompt = """You are an expert cryptocurrency trader with deep knowledge of technical analysis, market psychology, and risk management.

Your task is to analyze market data and make trading decisions (BUY, SELL, HOLD, or CLOSE positions).

Guidelines:
- Consider risk/reward ratio (minimum 2:1)
- Account for trading fees (~0.1% per trade)
- Never over-trade - quality over quantity
- Respect position sizing limits
- Always set stop-loss and take-profit levels
- Be conservative - only trade high-confidence setups

Analyze the provided context and respond ONLY with a valid JSON object in this exact format:
{
  "action": "BUY|SELL|HOLD|CLOSE",
  "confidence": 0.75,
  "size_usdt": 1000.0,
  "reasoning": "Brief explanation of your decision",
  "stop_loss": 62000.0,
  "take_profit": 65000.0
}

Rules:
- action: "BUY" to enter long, "SELL" to exit or short, "HOLD" to wait, "CLOSE" to close all positions
- confidence: 0.0 to 1.0 (only trade if >= 0.70)
- size_usdt: USD amount to trade (respect max_position_size)
- reasoning: 1-2 sentences explaining why
- stop_loss/take_profit: price levels (can be null for HOLD)

DO NOT include any text before or after the JSON. Output ONLY valid JSON.
"""
    
    async def analyze(
        self,
        symbol: str,
        market_data: Dict,
        portfolio: Dict,
        constraints: Dict
    ) -> Optional[Dict]:
        """
        Analyze market and make trading decision
        
        Args:
            symbol: Trading pair (e.g., "BTC/USDT")
            market_data: Current market state (price, indicators, candles)
            portfolio: Current portfolio state (cash, positions, equity)
            constraints: Trading constraints (max size, risk limits)
        
        Returns:
            Decision dict or None if error
        """
        try:
            # Build context
            context = {
                "symbol": symbol,
                "timestamp": datetime.now().isoformat(),
                "market": market_data,
                "portfolio": portfolio,
                "constraints": constraints,
            }
            
            # Create user message
            user_message = f"""Analyze this trading context and make a decision:

{json.dumps(context, indent=2)}

Remember: Respond ONLY with valid JSON, nothing else."""
            
            # Call Claude API
            logger.info(f"Requesting AI analysis for {symbol}")
            
            message = self.client.messages.create(
                model=self.model,
                max_tokens=1000,
                system=self.system_prompt,
                messages=[
                    {"role": "user", "content": user_message}
                ]
            )
            
            # Extract response
            response_text = message.content[0].text.strip()
            
            # Parse JSON response
            # Remove markdown code blocks if present
            if response_text.startswith("```"):
                # Extract JSON from code block
                lines = response_text.split("\n")
                response_text = "\n".join(lines[1:-1])
            
            decision = json.loads(response_text)
            
            # Validate decision structure
            required_fields = ['action', 'confidence', 'reasoning']
            if not all(field in decision for field in required_fields):
                logger.error(f"Invalid decision format: {decision}")
                return None
            
            # Add metadata
            decision['timestamp'] = datetime.now().isoformat()
            decision['symbol'] = symbol
            decision['context'] = context
            
            logger.info(
                f"AI decision for {symbol}: "
                f"{decision['action']} "
                f"(confidence: {decision['confidence']:.2f})"
            )
            
            return decision
            
        except json.JSONDecodeError as e:
            logger.error(f"Failed to parse AI response as JSON: {e}")
            logger.error(f"Response was: {response_text}")
            return None
        except Exception as e:
            logger.error(f"Error in AI analysis: {e}")
            return None
    
    def validate_decision(self, decision: Dict, portfolio: Dict) -> bool:
        """
        Validate that a decision meets all constraints
        """
        # Check confidence threshold
        if decision['confidence'] < settings.ai_min_confidence:
            logger.info(
                f"Decision rejected: confidence {decision['confidence']:.2f} "
                f"< minimum {settings.ai_min_confidence}"
            )
            return False
        
        # Check action is valid
        if decision['action'] not in ['BUY', 'SELL', 'HOLD', 'CLOSE']:
            logger.error(f"Invalid action: {decision['action']}")
            return False
        
        # For BUY actions, check we have enough cash
        if decision['action'] == 'BUY':
            size_usdt = decision.get('size_usdt', 0)
            if size_usdt > portfolio.get('cash_usdt', 0):
                logger.warning(
                    f"Insufficient cash: need {size_usdt}, "
                    f"have {portfolio.get('cash_usdt', 0)}"
                )
                return False
            
            # Check position size limit
            max_size = portfolio.get('equity_usdt', 0) * settings.max_position_size_pct
            if size_usdt > max_size:
                logger.warning(
                    f"Position size {size_usdt} exceeds max {max_size:.2f}"
                )
                return False
        
        return True


# Global instance
ai_strategy_service = AIStrategyService()
