"""
Execution Simulator - Paper trading with realistic fee/slippage simulation
"""
from typing import Dict, Optional
from decimal import Decimal
from datetime import datetime
import uuid
from sqlalchemy.orm import Session
from app.models import Order, Trade, Position
from app.config import settings
from app.services.portfolio import portfolio_service
import logging

logger = logging.getLogger(__name__)


class ExecutionSimulator:
    """
    Simulates order execution with realistic fees and slippage
    """
    
    def __init__(self):
        self.maker_fee = Decimal(str(settings.maker_fee))
        self.taker_fee = Decimal(str(settings.taker_fee))
        self.slippage_pct = Decimal(str(settings.slippage_pct))
    
    async def execute_market_order(
        self,
        db: Session,
        exchange: str,
        symbol: str,
        side: str,  # 'BUY' or 'SELL'
        quantity: float,
        current_price: float,
        ai_signal_id: Optional[int] = None
    ) -> Optional[Dict]:
        """
        Execute a market order in paper trading mode
        
        Returns:
            Execution result dict or None if failed
        """
        try:
            # Convert to Decimal for precision
            qty = Decimal(str(quantity))
            price = Decimal(str(current_price))
            
            # Apply slippage (worse price for market orders)
            if side == 'BUY':
                execution_price = price * (Decimal('1') + self.slippage_pct)
            else:
                execution_price = price * (Decimal('1') - self.slippage_pct)
            
            # Calculate notional value
            notional = qty * execution_price
            
            # Calculate fee (using taker fee for market orders)
            fee = notional * self.taker_fee
            
            # Generate order ID
            order_id = f"sim_{uuid.uuid4().hex[:12]}"
            
            # Create order record
            order = Order(
                order_id=order_id,
                exchange=exchange,
                symbol=symbol,
                side=side,
                type='market',
                quantity=qty,
                price=execution_price,
                status='filled',
                filled_quantity=qty,
                avg_fill_price=execution_price,
                fee=fee,
                ai_signal_id=ai_signal_id,
            )
            
            db.add(order)
            db.flush()
            
            # Process the trade based on side
            if side == 'BUY':
                pnl_realized = await self._process_buy(
                    db, exchange, symbol, qty, execution_price, fee
                )
            else:  # SELL
                pnl_realized = await self._process_sell(
                    db, exchange, symbol, qty, execution_price, fee
                )
            
            # Create trade record
            trade = Trade(
                exchange=exchange,
                symbol=symbol,
                side=side,
                type='market',
                quantity=qty,
                price=execution_price,
                fee=fee,
                fee_currency='USDT',
                pnl_realized=pnl_realized,
                order_id=order_id,
                ai_signal_id=ai_signal_id,
            )
            
            db.add(trade)
            db.commit()
            
            result = {
                'order_id': order_id,
                'symbol': symbol,
                'side': side,
                'quantity': float(qty),
                'price': float(execution_price),
                'fee': float(fee),
                'pnl_realized': float(pnl_realized) if pnl_realized else None,
                'status': 'filled',
                'timestamp': datetime.now().isoformat(),
            }
            
            logger.info(
                f"Executed {side} {float(qty)} {symbol} @ ${float(execution_price):.2f} "
                f"(fee: ${float(fee):.2f})"
            )
            
            return result
            
        except Exception as e:
            logger.error(f"Error executing order: {e}")
            db.rollback()
            return None
    
    async def _process_buy(
        self,
        db: Session,
        exchange: str,
        symbol: str,
        quantity: Decimal,
        price: Decimal,
        fee: Decimal
    ) -> Optional[Decimal]:
        """
        Process a BUY order - open or add to LONG position
        """
        # Calculate total cost (notional + fee)
        total_cost = (quantity * price) + fee
        
        # Check and update cash
        if total_cost > portfolio_service.cash_usdt:
            raise ValueError(f"Insufficient cash: need {total_cost}, have {portfolio_service.cash_usdt}")
        
        portfolio_service.cash_usdt -= total_cost
        
        # Check if position exists
        position = db.query(Position).filter(
            Position.symbol == symbol,
            Position.exchange == exchange
        ).first()
        
        if position and position.quantity > 0:
            # Add to existing position
            old_value = position.quantity * position.entry_price
            new_value = quantity * price
            total_quantity = position.quantity + quantity
            
            # Calculate weighted average entry price
            position.entry_price = (old_value + new_value) / total_quantity
            position.quantity = total_quantity
            position.current_price = price
            
            logger.info(f"Added to {symbol} position: new qty={float(total_quantity)}")
        else:
            # Create new LONG position
            position = Position(
                symbol=symbol,
                exchange=exchange,
                side='LONG',
                quantity=quantity,
                entry_price=price,
                current_price=price,
                leverage=1,
            )
            db.add(position)
            
            logger.info(f"Opened LONG position: {float(quantity)} {symbol} @ ${float(price):.2f}")
        
        return None  # No realized PnL on buy
    
    async def _process_sell(
        self,
        db: Session,
        exchange: str,
        symbol: str,
        quantity: Decimal,
        price: Decimal,
        fee: Decimal
    ) -> Optional[Decimal]:
        """
        Process a SELL order - close or reduce LONG position
        """
        # Find existing position
        position = db.query(Position).filter(
            Position.symbol == symbol,
            Position.exchange == exchange,
            Position.quantity > 0
        ).first()
        
        if not position:
            logger.warning(f"No position found for {symbol} to sell")
            return None
        
        if quantity > position.quantity:
            logger.warning(
                f"Sell quantity {float(quantity)} > position {float(position.quantity)}, "
                f"adjusting to position size"
            )
            quantity = position.quantity
        
        # Calculate realized PnL
        pnl_realized = (price - position.entry_price) * quantity - fee
        
        # Calculate proceeds (notional - fee)
        proceeds = (quantity * price) - fee
        
        # Update cash
        portfolio_service.cash_usdt += proceeds
        
        # Update position
        position.quantity -= quantity
        position.pnl_realized += pnl_realized
        
        if position.quantity == 0:
            logger.info(
                f"Closed {symbol} position: "
                f"PnL=${float(pnl_realized):.2f} "
                f"({float(pnl_realized / (position.entry_price * quantity) * 100):.2f}%)"
            )
            # Keep position record with 0 quantity for history
        else:
            logger.info(
                f"Reduced {symbol} position: "
                f"remaining={float(position.quantity)}, "
                f"realized PnL=${float(pnl_realized):.2f}"
            )
        
        return pnl_realized
    
    async def close_position(
        self,
        db: Session,
        symbol: str,
        current_price: float,
        ai_signal_id: Optional[int] = None
    ) -> Optional[Dict]:
        """
        Close an existing position completely
        """
        position = db.query(Position).filter(
            Position.symbol == symbol,
            Position.quantity > 0
        ).first()
        
        if not position:
            logger.warning(f"No position to close for {symbol}")
            return None
        
        # Execute SELL for entire position
        return await self.execute_market_order(
            db=db,
            exchange=position.exchange,
            symbol=symbol,
            side='SELL',
            quantity=float(position.quantity),
            current_price=current_price,
            ai_signal_id=ai_signal_id
        )


# Global instance
execution_simulator = ExecutionSimulator()
