"""
Technical Indicators - RSI, EMA, Volume analysis, etc.
"""
import pandas as pd
import numpy as np
from typing import List, Dict


def calculate_rsi(prices: List[float], period: int = 14) -> float:
    """
    Calculate Relative Strength Index
    
    Args:
        prices: List of closing prices (oldest first)
        period: RSI period (default 14)
    
    Returns:
        RSI value (0-100)
    """
    if len(prices) < period + 1:
        return 50.0  # Neutral if not enough data
    
    df = pd.DataFrame({'close': prices})
    
    # Calculate price changes
    delta = df['close'].diff()
    
    # Separate gains and losses
    gains = delta.where(delta > 0, 0)
    losses = -delta.where(delta < 0, 0)
    
    # Calculate average gains and losses
    avg_gains = gains.rolling(window=period).mean()
    avg_losses = losses.rolling(window=period).mean()
    
    # Calculate RS and RSI
    rs = avg_gains / avg_losses
    rsi = 100 - (100 / (1 + rs))
    
    return float(rsi.iloc[-1])


def calculate_ema(prices: List[float], period: int = 20) -> float:
    """
    Calculate Exponential Moving Average
    
    Args:
        prices: List of closing prices (oldest first)
        period: EMA period
    
    Returns:
        EMA value
    """
    if len(prices) < period:
        return float(np.mean(prices))  # SMA if not enough data
    
    df = pd.DataFrame({'close': prices})
    ema = df['close'].ewm(span=period, adjust=False).mean()
    
    return float(ema.iloc[-1])


def calculate_volatility(prices: List[float], period: int = 20) -> float:
    """
    Calculate historical volatility (standard deviation of returns)
    
    Args:
        prices: List of closing prices
        period: Lookback period
    
    Returns:
        Volatility as decimal (e.g., 0.02 = 2%)
    """
    if len(prices) < period:
        return 0.0
    
    df = pd.DataFrame({'close': prices[-period:]})
    returns = df['close'].pct_change().dropna()
    
    return float(returns.std())


def calculate_volume_ratio(volumes: List[float], period: int = 20) -> float:
    """
    Calculate current volume vs average volume ratio
    
    Args:
        volumes: List of volumes (oldest first)
        period: Period for average
    
    Returns:
        Volume ratio (e.g., 1.5 = 50% above average)
    """
    if len(volumes) < period + 1:
        return 1.0
    
    current_vol = volumes[-1]
    avg_vol = np.mean(volumes[-period-1:-1])
    
    if avg_vol == 0:
        return 1.0
    
    return float(current_vol / avg_vol)


def calculate_indicators(candles: List[List]) -> Dict:
    """
    Calculate all technical indicators from OHLCV data
    
    Args:
        candles: List of [timestamp, open, high, low, close, volume]
    
    Returns:
        Dict of indicators
    """
    if not candles or len(candles) < 20:
        return {
            'rsi_14': 50.0,
            'ema_20': 0.0,
            'ema_50': 0.0,
            'volatility': 0.0,
            'volume_ratio': 1.0,
            'price_change_pct': 0.0,
        }
    
    # Extract data
    closes = [c[4] for c in candles]
    volumes = [c[5] for c in candles]
    
    # Calculate indicators
    indicators = {
        'rsi_14': calculate_rsi(closes, period=14),
        'ema_20': calculate_ema(closes, period=20),
        'ema_50': calculate_ema(closes, period=50) if len(closes) >= 50 else calculate_ema(closes, period=20),
        'volatility': calculate_volatility(closes, period=20),
        'volume_ratio': calculate_volume_ratio(volumes, period=20),
        'price_change_pct': float((closes[-1] / closes[-2] - 1) * 100) if len(closes) >= 2 else 0.0,
    }
    
    # Add trend indicators
    if indicators['ema_20'] > 0 and indicators['ema_50'] > 0:
        indicators['trend'] = 'bullish' if indicators['ema_20'] > indicators['ema_50'] else 'bearish'
    else:
        indicators['trend'] = 'neutral'
    
    # RSI interpretation
    if indicators['rsi_14'] < 30:
        indicators['rsi_signal'] = 'oversold'
    elif indicators['rsi_14'] > 70:
        indicators['rsi_signal'] = 'overbought'
    else:
        indicators['rsi_signal'] = 'neutral'
    
    return indicators


def format_candles_for_analysis(candles: List[List], limit: int = 50) -> List[Dict]:
    """
    Format candles for AI analysis (last N candles)
    """
    recent_candles = candles[-limit:] if len(candles) > limit else candles
    
    return [
        {
            'timestamp': c[0],
            'open': c[1],
            'high': c[2],
            'low': c[3],
            'close': c[4],
            'volume': c[5],
        }
        for c in recent_candles
    ]
