import React, { useState, useEffect } from 'react'
import { TrendingUp, Activity, DollarSign, Bot, AlertCircle } from 'lucide-react'
import './App.css'

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:8000'
const WS_URL = import.meta.env.VITE_WS_URL || 'ws://localhost:8000/ws'

interface Portfolio {
  equity_usdt: number
  cash_usdt: number
  positions_value_usdt: number
  pnl_total: number
  pnl_pct: number
  num_positions: number
  num_trades: number
  positions: Position[]
}

interface Position {
  symbol: string
  side: string
  quantity: number
  entry_price: number
  current_price: number
  pnl_unrealized: number
  pnl_pct: number
}

interface Trade {
  id: number
  timestamp: string
  symbol: string
  side: string
  quantity: number
  price: number
  pnl_realized?: number
}

interface AISignal {
  timestamp: string
  symbol: string
  action: string
  confidence: number
  reasoning: string
}

function App() {
  const [portfolio, setPortfolio] = useState<Portfolio | null>(null)
  const [trades, setTrades] = useState<Trade[]>([])
  const [lastSignal, setLastSignal] = useState<AISignal | null>(null)
  const [connected, setConnected] = useState(false)
  const [loading, setLoading] = useState(true)

  // Fetch initial data
  useEffect(() => {
    fetchPortfolio()
    fetchTrades()

    // Setup WebSocket
    const ws = new WebSocket(WS_URL)

    ws.onopen = () => {
      console.log('WebSocket connected')
      setConnected(true)
    }

    ws.onmessage = (event) => {
      const data = JSON.parse(event.data)
      console.log('WebSocket message:', data)

      if (data.type === 'portfolio') {
        setPortfolio(data)
      } else if (data.type === 'ai_signal') {
        setLastSignal(data)
      } else if (data.type === 'order_filled') {
        fetchTrades() // Refresh trades
        fetchPortfolio() // Refresh portfolio
      }
    }

    ws.onclose = () => {
      console.log('WebSocket disconnected')
      setConnected(false)
    }

    return () => {
      ws.close()
    }
  }, [])

  const fetchPortfolio = async () => {
    try {
      const response = await fetch(`${API_URL}/api/portfolio`)
      const data = await response.json()
      setPortfolio(data)
      setLoading(false)
    } catch (error) {
      console.error('Error fetching portfolio:', error)
      setLoading(false)
    }
  }

  const fetchTrades = async () => {
    try {
      const response = await fetch(`${API_URL}/api/trades?limit=10`)
      const data = await response.json()
      setTrades(data)
    } catch (error) {
      console.error('Error fetching trades:', error)
    }
  }

  const triggerAnalysis = async () => {
    try {
      const response = await fetch(`${API_URL}/api/strategy/analyze?symbol=BTC/USDT`, {
        method: 'POST'
      })
      const data = await response.json()
      console.log('Analysis result:', data)
      alert('AI analysis triggered! Check the signal panel.')
    } catch (error) {
      console.error('Error triggering analysis:', error)
      alert('Error triggering analysis')
    }
  }

  if (loading) {
    return (
      <div className="loading-screen">
        <Activity size={48} className="spin" />
        <p>Loading Crypto AI Trader...</p>
      </div>
    )
  }

  return (
    <div className="app">
      <header className="header">
        <div className="header-left">
          <Bot size={32} />
          <h1>Crypto AI Trader</h1>
        </div>
        <div className="header-right">
          <div className={`status-indicator ${connected ? 'connected' : 'disconnected'}`}>
            <div className="status-dot"></div>
            {connected ? 'Connected' : 'Disconnected'}
          </div>
        </div>
      </header>

      <main className="main">
        {/* Portfolio Summary */}
        <div className="grid">
          <div className="card">
            <div className="card-header">
              <DollarSign size={20} />
              <span>Total Equity</span>
            </div>
            <div className="card-value">
              ${portfolio?.equity_usdt.toFixed(2) || '0.00'}
            </div>
            <div className="card-subtitle">
              Cash: ${portfolio?.cash_usdt.toFixed(2) || '0.00'}
            </div>
          </div>

          <div className="card">
            <div className="card-header">
              <TrendingUp size={20} />
              <span>Total P&L</span>
            </div>
            <div className={`card-value ${(portfolio?.pnl_pct || 0) >= 0 ? 'positive' : 'negative'}`}>
              {(portfolio?.pnl_pct || 0) >= 0 ? '+' : ''}
              {portfolio?.pnl_pct.toFixed(2) || '0.00'}%
            </div>
            <div className="card-subtitle">
              ${portfolio?.pnl_total.toFixed(2) || '0.00'}
            </div>
          </div>

          <div className="card">
            <div className="card-header">
              <Activity size={20} />
              <span>Positions</span>
            </div>
            <div className="card-value">
              {portfolio?.num_positions || 0}
            </div>
            <div className="card-subtitle">
              {portfolio?.num_trades || 0} trades total
            </div>
          </div>

          <div className="card">
            <div className="card-header">
              <Bot size={20} />
              <span>AI Status</span>
            </div>
            <div className="card-value ai-status">
              {lastSignal ? lastSignal.action : 'WAITING'}
            </div>
            <div className="card-subtitle">
              {lastSignal ? `${(lastSignal.confidence * 100).toFixed(0)}% confident` : 'Analyzing...'}
            </div>
          </div>
        </div>

        {/* AI Signal Panel */}
        {lastSignal && (
          <div className="card ai-signal-card">
            <div className="card-header">
              <Bot size={20} />
              <span>Latest AI Signal</span>
              <span className="timestamp">
                {new Date(lastSignal.timestamp).toLocaleTimeString()}
              </span>
            </div>
            <div className="ai-signal-content">
              <div className="ai-signal-action">
                <span className={`action-badge ${lastSignal.action.toLowerCase()}`}>
                  {lastSignal.action}
                </span>
                <span className="symbol">{lastSignal.symbol}</span>
              </div>
              <div className="ai-reasoning">
                <AlertCircle size={16} />
                <span>{lastSignal.reasoning}</span>
              </div>
              <div className="confidence-bar">
                <div 
                  className="confidence-fill" 
                  style={{ width: `${lastSignal.confidence * 100}%` }}
                ></div>
              </div>
            </div>
          </div>
        )}

        {/* Open Positions */}
        {portfolio && portfolio.positions.length > 0 && (
          <div className="card">
            <div className="card-header">
              <TrendingUp size={20} />
              <span>Open Positions</span>
            </div>
            <div className="table-container">
              <table className="data-table">
                <thead>
                  <tr>
                    <th>Symbol</th>
                    <th>Side</th>
                    <th>Quantity</th>
                    <th>Entry</th>
                    <th>Current</th>
                    <th>P&L</th>
                  </tr>
                </thead>
                <tbody>
                  {portfolio.positions.map((pos, idx) => (
                    <tr key={idx}>
                      <td className="symbol-cell">{pos.symbol}</td>
                      <td>
                        <span className={`badge ${pos.side.toLowerCase()}`}>
                          {pos.side}
                        </span>
                      </td>
                      <td>{pos.quantity.toFixed(6)}</td>
                      <td>${pos.entry_price.toFixed(2)}</td>
                      <td>${pos.current_price.toFixed(2)}</td>
                      <td className={pos.pnl_pct >= 0 ? 'positive' : 'negative'}>
                        {pos.pnl_pct >= 0 ? '+' : ''}
                        {pos.pnl_pct.toFixed(2)}%
                        <div className="subtitle">
                          ${pos.pnl_unrealized.toFixed(2)}
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}

        {/* Recent Trades */}
        <div className="card">
          <div className="card-header">
            <Activity size={20} />
            <span>Recent Trades</span>
          </div>
          <div className="table-container">
            <table className="data-table">
              <thead>
                <tr>
                  <th>Time</th>
                  <th>Symbol</th>
                  <th>Side</th>
                  <th>Quantity</th>
                  <th>Price</th>
                  <th>P&L</th>
                </tr>
              </thead>
              <tbody>
                {trades.length === 0 ? (
                  <tr>
                    <td colSpan={6} className="empty-state">
                      No trades yet. AI will start trading soon!
                    </td>
                  </tr>
                ) : (
                  trades.map((trade) => (
                    <tr key={trade.id}>
                      <td>{new Date(trade.timestamp).toLocaleTimeString()}</td>
                      <td className="symbol-cell">{trade.symbol}</td>
                      <td>
                        <span className={`badge ${trade.side.toLowerCase()}`}>
                          {trade.side}
                        </span>
                      </td>
                      <td>{trade.quantity.toFixed(6)}</td>
                      <td>${trade.price.toFixed(2)}</td>
                      <td className={trade.pnl_realized && trade.pnl_realized >= 0 ? 'positive' : 'negative'}>
                        {trade.pnl_realized 
                          ? `${trade.pnl_realized >= 0 ? '+' : ''}$${trade.pnl_realized.toFixed(2)}`
                          : '-'
                        }
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>

        {/* Controls */}
        <div className="controls">
          <button className="btn btn-primary" onClick={triggerAnalysis}>
            <Bot size={20} />
            Trigger AI Analysis
          </button>
          <button className="btn btn-secondary" onClick={fetchPortfolio}>
            <Activity size={20} />
            Refresh
          </button>
        </div>
      </main>

      <footer className="footer">
        <p>
          ⚠️ Paper Trading Mode | Educational Purposes Only | Not Financial Advice
        </p>
      </footer>
    </div>
  )
}

export default App
